' Assembler for SAP1 Computer
' Rev 1.0.0 William M Leue 9-Apr-2022

option default integer
option base 0

' Constants

' Microcode ROM sizes and indices
const NBITS  = 8
const RSIZE  = 16
const NOPS   = 16
const LDIMAX = 15

' Assembler Constants
const N_DIRECTIVES = 2
const SYMTAB_LEN = 1000
const MAX_PROGRAMS = 30
const MAX_LINES  = 100

' Opcodes
const opNOP = 0
const opLDA = 1
const opADD = 2
const opSUB = 3
const opSTA = 4
const opLDI = 5
const opJMP = 6
const opJZ  = 7
const opJC  = 8
const opJN  = 9
const opOUT = 14
const opHLT = 15

' Keyboard codes
const UP    = 128
const DOWN  = 129
const ENTER = 13
const ESC   = 27

' Globals

' simulated RAM
dim ram(RSIZE)

' Program Load and Assembler
dim source$(MAX_LINES)
dim nlines = 0
dim assoc_address(MAX_LINES)
dim op_values(NOPS)
dim op_names$(NOPS)
dim as_dnames$(N_DIRECTIVES)
dim as_mnames$(NOPS)
dim pselect = 1
dim nprograms = 0
dim program_names$(MAX_PROGRAMS)
dim address = 0
dim symtab$(3, SYMTAB_LEN)
dim nsymbols = 0
dim astyps(MAX_LINES)
dim aslbls$(MAX_LINES)
dim asops(MAX_LINES)
dim asvals$(MAX_LINES)
dim asadrs(MAX_LINES)
dim asdirs(MAX_LINES)

' Main Program
open "debug.txt" for output as #1
ReadAssemblerResources
DoChooseProgram
DoAssembly
end

' Read assembler resources
sub ReadAssemblerResources
  local i
  for i = 0 to NOPS-1
    read op_names$(i)
  next i
  for i = 0 to NOPS-1
    read op_values(i) 
  next i
  for i = 1 to N_DIRECTIVES
    read as_dnames$(i)
  next i
end sub

' Choose a progam (.asm) file
sub DoChooseProgram
  local x, y
  pmode = 2
  DrawProgramlist
  pselect = 1
  HandleKeyboardEvents
end sub

' Print a list of available program source files (Programs/*.asm)
sub DrawProgramList
  local f$
  local x, y, i
  nprograms = 0
  f$ = dir$("Programs/*.asm", FILE)
  do while f$ <> ""
    inc nprograms
    program_names$(nprograms) = f$
    f$ = dir$()
  loop
  box 50, 50, 350, 450,, rgb(white), rgb(black)
  x = 55 : y = 60
  for i = 1 to nprograms
    if i = pselect then
      text x, y, program_names$(i), "LT", 7,, rgb(white), rgb(blue)
    else
      text x, y, program_names$(i), "LT", 7,, rgb(white), -1
    end if
    inc y, 20
  next i
end sub

' Process keystrokes for navigating the program list
sub HandleKeyboardEvents
  local z$
  local cmd
  z$ = INKEY$
  do
    do
      z$ = INKEY$
    loop until z$ <> ""
    cmd = asc(z$)
    select case cmd
      case UP
        if pselect > 1 then
          inc pselect, -1
          DrawProgramList
        end if
      case DOWN
        if pselect < nprograms then
          inc pselect
          DrawProgramList
        end if
      case ENTER
        exit do
      case ESC
        pselect = 0
        end
    end select
  loop
end sub

' Assemble the code
' Pass 1: capture and store symbolic labels and their values
' Pass 2: assemble
sub DoAssembly
  local pname$
  local pl
  ClearMemory
  InitSymtab
  d$ = " " + chr$(9)
  pname$ = "Programs/" + program_names$(pselect)
  pl = len(pname$)
  ReadSource pname$
  ClearAssociatedAddresses
  AssemblerPass1x
  AssemblerPass2x
  PrettyPrintCode
  pname$ = LEFT$(pname$, pl-4) + ".bin"
  PrintBinary pname$
  ShowStatus "Successful Assembly"
end sub

' Read the source file into a memory buffer
sub ReadSource path$
  local buf$
  on error skip 1
  open path$ for input as #2
  if mm.errno <> 0 then
    ShowStatus "Error opening source file '" + path$ + "' for input"
    exit sub
  end if
  nlines = 0
  do
    inc nlines
    line input #2, source$(nlines)
  loop until eof(#2)
  close #2
end sub

' Clear the list of ram addresses associated with lines of source code.
sub ClearAssociatedAddresses
  local i
  for i = 1 to MAX_LINES
    assoc_address(i) = -1
  next i
end sub

' Parse the source code. For each line:
'  1. determine the statement type (0..5)
'  2. For each component, determine the type and value
'  3. Make arrays of this info.
sub AssemblerPass1x
  local buf$, p$, d$, f2$, f3$, f4$, s$
  local cfx, dx, lnum, address, t
  d$ = " " + chr$(9)
  lnum = 0
  address = 0
  do
    inc lnum
    buf$ = source$(lnum)
    p$ = AsmField$(buf$, d$)
    cfx = CommentFieldIndex(p$)
    t = GetStatementType(p$, cfx)
    astyps(lnum) = t
    if t > 1 then
      if address > RSIZE then
        ShowStatus "Error -- address larger than max RAM"
        exit sub
      end if
      if t <> 5 then assoc_address(lnum) = address
      f2$ = field$(p$, 2, "/")
      f3$ = field$(p$, 3, "/")
      f4$ = field$(p$, 4, "/")    
      if len(f2$) > 0 and instr(f2$, ":") = len(f2$) then
        s$ = LEFT$(f2$, len(f2$)-1)
        AddSymbol s$, f4$, str$(address)
      end if
    end if
    select case t
      case 2
        aslbls$(lnum) = ""
        asops(lnum) = op_values(FindOpcode(f3$))
print #1, "f3$: ";f3$;" asops(lnum): ";asops(lnum)
        asadrs(lnum) = address
        if len(f4$) > 0 then
          asvals$(lnum) = f4$
        else
          asvals$(lnum) = "0"
        end if
        inc address
      case 3
        aslbls$(lnum) = f2$
        asops(lnum) = op_values(FindOpcode(f3$))
        if len(f4$) > 0 then
          asvals$(lnum) = f4$
        else
          asvals$(lnum) = "0"
        end if
        asadrs(lnum) = address
        inc address
      case 4
        aslbls$(lnum) = f2$
        asvals$(lnum) = f3$
        asadrs(lnum) = address
        inc address
      case 5
        asdirs(lnum) = FindDirective(f3$)
        asvals$(lnum) = f4$
        address = val(f4$)
        asadrs(lnum) = address
    end select        
  loop until lnum = nlines
  'PrintStatementTypes
end sub

' Print the Statement type into after Pass1 (debug)
sub PrintStatementTypes
  local i, lbl
  print #1, "Statement Type Info:"
  print #1, "Line Type Label Address Dir Op   Value"
  for i = 1 to nlines
    print #1, " ";format$(i, "%2g");"  ";astyps(i);space$(4);
    if astyps(i) > 1 then
      lbl = len(aslbls$(i))
      print #1, aslbls$(i);space$(6-lbl);bin$(asadrs(i),4);
      if astyps(i) = 5 then
        print #1, "   ";asdirs(i);"        ";asvals$(i)
      else if astyps(i) = 2 or astyps(i) = 3 then
        print #1,"       ";bin$(asops(i), 4);"  ";asvals$(i)
      end if
      if astyps(i) = 4 then print #1, "             "; asvals$(i)
    else
      print #1, ""
    end if
  next i
end sub

' Do the actual code assembly and create the ram image
' The only real job is to look up symbolic values in the
' symbol table.
sub AssemblerPass2x
  local i, t, opcode, operand$, addr, v$, a$, vopand
  for i = 1 to nlines
    t = astyps(i)
    if t > 1 and t < 5 then
      opcode = asops(i)
      operand$ = asvals$(i)
      if opcode = opOUT or opcode = opHLT then
        operand$ = "0"
      end if
      addr = asadrs(i)
      if not IsNumeric(operand$) then
        GetSymbolValue operand$, v$, a$
        vopand = val(a$)
      else
        vopand = val(operand$)
        if opcode = opLDI and vopand > LDIMAX then
          ShowStatus "Operand of LDI too large (max = 15)"
          exit sub
        end if
      end if
      ram(addr) = (opcode << 4) or vopand
    end if
  next i
end sub

' returns 1 if a string is decimal numeric, 0 otherwise
function IsNumeric(operand$)
  local i, ol, c$, ok
  ok = 1
  ol = len(operand$)
  for i = 1 to ol
    c$ = MID$(operand$, i, 1)
    if asc(c$) < asc("0") or asc(c$) > asc("9") then
      ok = 0
      exit for
    end if
  next i
  IsNumeric = ok
end function

' Get the statement type for a line of code:
' Type 0:  null line
' Type 1:  just a comment
' Type 2:  <operation>  <operand>
' Type 3:  <label>: <operation> <operand>
' Type 4:  <label>:
' Type 5:  <label>: <value>
' Type 6:  .<directive> <value>
' Arguments:
'  p$: pre-parsed statement line
'  cfx: comment delimiter location in line (from 1)
function GetStatementType(p$, cfx)
  local i, lflag, dflag, oflag, vflag
  local pf$(4)
  if len(p$) = 0 then
    GetStatementType = 0
    exit function
  end if
  if cfx = 1 then
    GetStatementType = 1
    exit function
  end if
  for i = 2 to 4
    pf$(i) = field$(p$, i, "/")
  next i
  lflag = 0 : dflag = 0 : oflag = 0
  if len(pf$(2)) > 0 and instr(pf$(2), ":") > 0 then
    lflag = 1
  end if
  if len(pf$(3)) > 0 then
    if FindDirective(pf$(3)) > 0 then
      dflag = 1
    else if FindOpcode(pf$(3)) > 0 then
      oflag = 1
    end if
  end if
  vflag = 0
  if len(pf$(4)) > 0 then
    vflag = 1
  end if
  if lflag then
    if oflag then
      GetStatementType = 3
    else
      GetStatementType = 4
    end if
  else if oflag then  
    GetStatementType = 2
  else
    if dflag then
      if vflag then
        GetStatementType = 5
      else
        GetStatementType = -1
      end if
    end if
  end if
end function

' Print the binary ram image to the '.bin' file
sub PrintBinary bpath$
  local i
  on error skip 1
  open bpath$ for output as #2
  if mm.errno <> 0 then
    print "Error opening '" + bpath$ + "' for output"
    exit sub
  end if
  for i = 0 to RSIZE-1
    print #2, bin$(ram(i), 8)
  next i
  close #2
end sub

' Print out the source code and assembled RAM contents
sub PrettyPrintCode
  local lnum, addr, sl
  cls
  for i = 1 to nlines
    sl = len(source$(i))
    print source$(i);
    addr = assoc_address(i)
    if addr >= 0 then
      print space$(15-sl);
      print bin$(addr, 4) + " ";
      print bin$(ram(addr), 8)
    else
      print ""
    end if
  next i
end sub

' Print out status messages
sub ShowStatus m$
  print m$
end sub

' Look up an assembler directive (starting with a .)
function FindDirective(f$)
  local i
  for i = 0 to N_DIRECTIVES-1
    if UCASE$(f$) = as_dnames$(i) then
      FindDirective = i
      exit function
    end if
  next i
  ShowStatus "Error - Unknown Directive '" + f$ + "'"
  FindDirective = -1
end function

' Look up an opcode value
function FindOpcode(f$)
  local i
  for i = 0 to NOPS-1
    if UCASE$(f$) = op_names$(i) then
      FindOpcode = op_values(i)
      exit function
    end if
  next i
  ShowStatus "Error - Unknown OpCode '" + f$ + "'"
  FindOpcode = -1      
end function

' Parser for an Assembler: replace all delimiter fields with '/';
' semicolon and colon get their own fields.
function AsmField$(s$, d$)
  local p, b, t
  local c$, f$, tf$
  t = 0
  f$ = ""
  tf$ = ""
  for p = 1 to len(s$)
    c$ = MID$(s$, p, 1)
    b = instr(1, d$, c$)
    if t = 0 then
      if b = 0 then
        t = 1
        tf$ = "/" + c$
      else
        t = 2
        f$ = f$ + "//"
      end if
    else if t = 1 then
      if b > 0 then
        t = 2
        f$ = f$ + tf$ + "/"
      else
        if c$ = ";" then
          if len(tf$) > 0 then
            f$ = f$ + tf$
            tf$ = ""
          end if
          f$ = f$ + "/;/"
          tf$ = ""
        else
          tf$ = tf$ + c$
        end if
      end if
    else
      if b = 0 then
        if c$ = ";" then
          f$ = f$ + ";/"
          tf$ = ""
        else
          t = 1
          tf$ = c$
        end if
      end if
    end if      
  next p
  if t = 1 then
    f$ = f$ + tf$ + "/"
  end if
  AsmField$ = f$
end function  

' Function to return the field index (1's based) of the comment
' delimiter ';'. Returns -1 if no delimter is found. The input
' string MUST be the output string of the AsmField() function.
function CommentFieldIndex(f$)
  local p, ix
  local c$
  ix = 0
  if len(f$) = 0 then
    CommentFieldIndex = -1
    exit function
  end if
  for p = 1 to len(f$)
    c$ = MID$(f$, p, 1)
    if c$ = "/" then
      inc ix
    end if
    if c$ = ";" then
      CommentFieldIndex = ix
      exit function
    end if
  next p
  CommentFieldIndex = -1
end function

' (debug) print memory contents
sub PrintMemory
  local i
  print #1, "Memory Contents"
  for i = 0 to RSIZE-1
    print #1, i;": ";bin$(ram(i), 8)
  next i
end sub

' Clear memory
sub ClearMemory
  local i
  for i = 1 to RSIZE
    ram(i) = 0
  next i
end sub

' Initialize assembler symbol table
sub InitSymtab
  local i, j
  for i = 1 to SYMTAB_LEN
    for j = 1 to 3
      symtab$(j, i) = ""
    next j
  next i
  nsymbols = 0
end sub

' Add a symbol to the symbol table
sub AddSymbol s$, v$, a$
'print #1, "AddSymbol ";s$;", ";v$;", ";a$
  inc nsymbols
  symtab$(1, nsymbols) = s$
  symtab$(2, nsymbols) = v$
  symtab$(3, nsymbols) = a$
end sub

' Look up a symbol value in the symbol table
sub GetSymbolValue s$, v$, a$
  local i
  v$ = ""
  a$ = ""
  for i = 1 to nsymbols
    if symtab$(1, i) = s$ then
      hit = 1
      v$ = symtab$(2, i)
      a$ = symtab$(3, i)
      exit sub
    end if
  next i
  print "Error - Cannot find value for symbol '";s$;"'"
  end
end sub

' Only data statements beyond this point!

' Opcode Mnemonics (for all 16 slots)
data "NOP", "LDA", "ADD", "SUB", "STA", "LDI", "J", "JC", "JZ"
data "JN", "NOP", "NOP", "NOP", "NOP", "OUT", "HLT"

' Opcode values for all 16 slots
data &B0000, &B0001, &B0010, &B0011, &B0100, &B0101, &B0110, &B0111
data &B1000, &B1001, &B0000, &B0000, &B0000, &B0000, &B1110, &B1111

' Assembler Directives
data ".ORG", ".WORD"


